/*
 *  $Id: distance-transform.c 28903 2025-11-24 15:49:51Z yeti-dn $
 *  Copyright (C) 2003-2025 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include <glib/gi18n-lib.h>

#include "libgwyddion/macros.h"
#include "libgwyddion/math.h"
#include "libgwyddion/arithmetic.h"
#include "libgwyddion/grains-nield.h"
#include "libgwyddion/distance-transform.h"

#include "libgwyddion/int-list.h"
#include "libgwyddion/internal.h"

enum {
    SEDINF = 0x7fffffffu,
    QUEUED = 0x80000000u,
};

typedef struct {
    gdouble distance;
    guint i;
    guint j;
} DistantPoint;

typedef struct {
    gdouble dist, ndist;
    gint j, i;
} ThinCandidate;

typedef gboolean (*ErodeFunc)(guint *grain,
                              gint width, gint height,
                              guint id,
                              const PixelQueue *inqueue,
                              PixelQueue *outqueue);


/* Euclidean distance transform */

static inline void
pixel_queue_add(PixelQueue *queue,
                gint i, gint j)
{
    if (G_UNLIKELY(queue->len == queue->size)) {
        queue->size = MAX(2*queue->size, 16);
        queue->points = g_renew(GridPoint, queue->points, queue->size);
    }

    queue->points[queue->len].i = i;
    queue->points[queue->len].j = j;
    queue->len++;
}

// Set squared distance for all points that have an 8-neighbour outside and add them to the queue.
static void
distance_transform_first_step(guint *distances,
                              guint xres, guint yres,
                              IntList *queue,
                              gboolean from_border)
{
    gsize k = 0;

    queue->len = 0;
    for (guint i = 0; i < yres; i++) {
        gboolean first_row = (i == 0);
        gboolean last_row = (i == yres-1);

        for (guint j = 0; j < xres; j++, k++) {
            gboolean first_column = (j == 0);
            gboolean last_column = (j == xres-1);

            if (!distances[k])
                continue;

            if ((from_border && (first_row || first_column || last_row || last_column))
                || (!first_row && !distances[k-xres])
                || (!first_column && !distances[k-1])
                || (!last_column && !distances[k+1])
                || (!last_row && !distances[k+xres])) {
                distances[k] = 1;
                int_list_add(queue, k);
            }
            else if ((!first_row && !first_column && !distances[k-xres-1])
                     || (!first_row && !last_column && !distances[k-xres+1])
                     || (!last_row && !first_column && !distances[k+xres-1])
                     || (!last_row && !last_column && !distances[k+xres+1])) {
                distances[k] = 2;
                int_list_add(queue, k);
            }
        }
    }
}

static void
distance_transform_erode_sed(guint *distances, const guint *olddist,
                             guint xres, guint yres,
                             guint l,
                             const IntList *inqueue,
                             IntList *outqueue)
{
    guint hvsed2 = 2*l - 1, diag2 = 2*hvsed2;
    guint q;

    outqueue->len = 0;

    for (q = 0; q < inqueue->len; q++) {
        guint k = inqueue->data[q], kk = k-xres-1;
        guint i = k/xres, j = k % xres;
        gboolean first_row = (i == 0);
        gboolean last_row = (i == yres-1);
        gboolean first_column = (j == 0);
        gboolean last_column = (j == xres-1);
        guint d2hv = olddist[k] + hvsed2, d2d = olddist[k] + diag2;

        if (!first_row && !first_column && (distances[kk] & ~QUEUED) > d2d) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2d;
        }
        kk++;
        if (!first_row && (distances[kk] & ~QUEUED) > d2hv) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2hv;
        }
        kk++;
        if (!first_row && !last_column && (distances[kk] & ~QUEUED) > d2d) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2d;
        }
        kk += xres-2;
        if (!first_column && (distances[kk] & ~QUEUED) > d2hv) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2hv;
        }
        kk += 2;
        if (!last_column && (distances[kk] & ~QUEUED) > d2hv) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2hv;
        }
        kk += xres-2;
        if (!last_row && !first_column && (distances[kk] & ~QUEUED) > d2d) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2d;
        }
        kk++;
        if (!last_row && (distances[kk] & ~QUEUED) > d2hv) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2hv;
        }
        kk++;
        if (!last_row && !last_column && (distances[kk] & ~QUEUED) > d2d) {
            if (!(distances[kk] & QUEUED))
                int_list_add(outqueue, kk);
            distances[kk] = QUEUED | d2d;
        }
    }
}

/**
 * distance_transform_raw:
 * @distances: Array @xres*@yres with nonzeros within shapes.  If all non-zero values are %SEDINF you can pass %TRUE
 *             for @infinitised.
 * @workspace: Workspace of identical dimensions as @distances.
 * @xres: Width.
 * @yres: Height.
 * @inqueue: Pre-allocated queue used by the algorithm.
 * @outqueue: Second pre-allocated queue used by the algorithm.
 * @from_border: %TRUE to consider image edges to be grain boundaries.
 *
 * Performs distance transformation.
 *
 * When it finishes non-zero values in @distances are squared Euclidean distances from outside pixels (including
 * outside the field).
 *
 * Workspace objects @workspace, @inqueue and @outqueue do not carry any information.  They are allocated by the
 * caller to enable an efficient repeated use.
 **/
static void
distance_transform_raw(guint *distances, guint *workspace,
                       guint xres, guint yres,
                       IntList *inqueue, IntList *outqueue,
                       gboolean from_border)
{
    guint l, q;

    distance_transform_first_step(distances, xres, yres, inqueue, from_border);

    for (l = 2; inqueue->len; l++) {
        gssize *qdata;

        for (q = 0; q < inqueue->len; q++) {
            gsize k = inqueue->data[q];
            workspace[k] = distances[k];
        }
        distance_transform_erode_sed(distances, workspace, xres, yres, l, inqueue, outqueue);

        qdata = outqueue->data;
        for (q = outqueue->len; q; q--, qdata++)
            distances[*qdata] &= ~QUEUED;

        GWY_SWAP(IntList*, inqueue, outqueue);
    }
}

static void
euclidean_distance_transform(GwyNield *nield, GwyField *distances, gboolean from_border)
{
    g_return_if_fail(GWY_IS_NIELD(nield));
    g_return_if_fail(GWY_IS_FIELD(distances));

    gint xres = nield->xres, yres = nield->yres;
    gsize n = (gsize)xres * (gsize)yres;
    const gint *d = nield->priv->data;
    guint *dist = g_new(guint, n);
    guint *workspace = g_new(guint, n);

    for (gsize k = 0; k < n; k++)
        dist[k] = (d[k] > 0) ? SEDINF : 0;

    guint inisize = (guint)(8*sqrt(n) + 16);
    IntList *inqueue = int_list_new(inisize);
    IntList *outqueue = int_list_new(inisize);

    distance_transform_raw(dist, workspace, xres, yres, inqueue, outqueue, from_border);

    int_list_free(inqueue);
    int_list_free(outqueue);

    gdouble *f = distances->priv->data;
    for (gsize k = 0; k < n; k++)
        f[k] = sqrt(dist[k]);
    gwy_field_invalidate(distances);

    g_free(workspace);
    g_free(dist);
}

/* Init @queue with all Von Neumann-neighbourhood boundary pixels. */
static void
init_erosion_4(guint *grain,
               guint width, guint height,
               gboolean from_border,
               PixelQueue *queue)
{
    queue->len = 0;
    for (guint i = 0; i < height; i++) {
        gboolean ifirst = !i, ilast = (i == height-1);
        for (guint j = 0; j < width; j++) {
            gboolean jfirst = !j, jlast = (j == width-1);
            guint k = i*width + j;
            if (!grain[k])
                continue;

            if ((!ifirst && !grain[k - width])
                || (!jfirst && !grain[k-1])
                || (!jlast && !grain[k+1])
                || (!ilast && !grain[k + width])
                || (from_border && (ifirst || ilast || jfirst || jlast))) {
                grain[k] = 1;
                pixel_queue_add(queue, i, j);
            }
        }
    }
}

/* Init @queue with all Von Neumann-neighbourhood boundary pixels. */
static void
init_erosion_8(guint *grain,
               gint width, gint height,
               gboolean from_border,
               PixelQueue *queue)
{
    queue->len = 0;
    for (guint i = 0; i < height; i++) {
        gboolean ifirst = !i, ilast = (i == height-1);
        for (guint j = 0; j < width; j++) {
            gboolean jfirst = !j, jlast = (j == width-1);
            guint k = i*width + j;
            if (!grain[k])
                continue;

            if ((!ifirst && !jfirst && !grain[k-1 - width])
                || (!ifirst && !grain[k - width])
                || (!ifirst && !jlast && !grain[k+1 - width])
                || (!jfirst && !grain[k-1])
                || (!jlast && !grain[k+1])
                || (!ilast && !jfirst && !grain[k-1 + width])
                || (!ilast && !grain[k + width])
                || (!ilast && !jlast && !grain[k+1 + width])
                || (from_border && (ifirst || ilast || jfirst || jlast))) {
                grain[k] = 1;
                pixel_queue_add(queue, i, j);
            }
        }
    }
}

static gboolean
erode_4(guint *grain,
        gint width, gint height,
        guint id,
        const PixelQueue *inqueue,
        PixelQueue *outqueue)
{
    const GridPoint *ipt = inqueue->points;
    guint m;

    outqueue->len = 0;
    for (m = inqueue->len; m; m--, ipt++) {
        gint i = ipt->i, j = ipt->j, k = i*width + j;

        if (i && grain[k - width] == G_MAXUINT) {
            grain[k - width] = id+1;
            pixel_queue_add(outqueue, i-1, j);
        }
        if (j && grain[k - 1] == G_MAXUINT) {
            grain[k - 1] = id+1;
            pixel_queue_add(outqueue, i, j-1);
        }
        if (j < width-1 && grain[k + 1] == G_MAXUINT) {
            grain[k + 1] = id+1;
            pixel_queue_add(outqueue, i, j+1);
        }
        if (i < height-1 && grain[k + width] == G_MAXUINT) {
            grain[k + width] = id+1;
            pixel_queue_add(outqueue, i+1, j);
        }
    }

    return outqueue->len;
}

static gboolean
erode_8(guint *grain,
        gint width, gint height,
        guint id,
        const PixelQueue *inqueue,
        PixelQueue *outqueue)
{
    const GridPoint *ipt = inqueue->points;
    guint m;

    outqueue->len = 0;
    for (m = inqueue->len; m; m--, ipt++) {
        gint i = ipt->i, j = ipt->j, k = i*width + j;
        if (i && j && grain[k - width - 1] == G_MAXUINT) {
            grain[k - width - 1] = id+1;
            pixel_queue_add(outqueue, i-1, j-1);
        }
        if (i && grain[k - width] == G_MAXUINT) {
            grain[k - width] = id+1;
            pixel_queue_add(outqueue, i-1, j);
        }
        if (i && j < width-1 && grain[k - width + 1] == G_MAXUINT) {
            grain[k - width + 1] = id+1;
            pixel_queue_add(outqueue, i-1, j+1);
        }
        if (j && grain[k - 1] == G_MAXUINT) {
            grain[k - 1] = id+1;
            pixel_queue_add(outqueue, i, j-1);
        }
        if (j < width-1 && grain[k + 1] == G_MAXUINT) {
            grain[k + 1] = id+1;
            pixel_queue_add(outqueue, i, j+1);
        }
        if (i < height-1 && j && grain[k + width - 1] == G_MAXUINT) {
            grain[k + width - 1] = id+1;
            pixel_queue_add(outqueue, i+1, j-1);
        }
        if (i < height-1 && grain[k + width] == G_MAXUINT) {
            grain[k + width] = id+1;
            pixel_queue_add(outqueue, i+1, j);
        }
        if (i < height-1 && j < width-1 && grain[k + width + 1] == G_MAXUINT) {
            grain[k + width + 1] = id+1;
            pixel_queue_add(outqueue, i+1, j+1);
        }
    }

    return outqueue->len;
}

/* Perform a cityblock, chessboard or octagonal distance transform of given type using provided queues. */
/* TODO: Change the argument to Nield (which includes the dimensions). */
guint
_gwy_simple_dist_trans(guint *grain, guint width, guint height,
                       gboolean from_border, GwyDistanceTransformType dtype,
                       PixelQueue *inqueue, PixelQueue *outqueue)
{
    ErodeFunc erode = NULL;
    guint dist = 1;

    inqueue->len = outqueue->len = 0;

    if (dtype == GWY_DISTANCE_TRANSFORM_CONN4 || dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL48)
        init_erosion_4(grain, width, height, from_border, inqueue);
    else if (dtype == GWY_DISTANCE_TRANSFORM_CONN8 || dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL84)
        init_erosion_8(grain, width, height, from_border, inqueue);

    if (dtype == GWY_DISTANCE_TRANSFORM_CONN4 || dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL84)
        erode = erode_4;
    else if (dtype == GWY_DISTANCE_TRANSFORM_CONN8 || dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL48)
        erode = erode_8;

    g_return_val_if_fail(erode, 0);

    while (TRUE) {
        if (!erode(grain, width, height, dist, inqueue, outqueue))
            break;
        GWY_SWAP(PixelQueue*, inqueue, outqueue);
        dist++;

        if (dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL48 || dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL84)
            erode = (erode == erode_4) ? erode_8 : erode_4;
    }

    return dist;
}

static void
average_octagonal_dt(GwyNield *nield, GwyField *distances, gboolean from_border)
{
    GwyField *tmp = gwy_field_new_alike(distances, FALSE);
    gwy_nield_distance_transform(nield, distances, GWY_DISTANCE_TRANSFORM_OCTAGONAL48, from_border);
    gwy_nield_distance_transform(nield, tmp, GWY_DISTANCE_TRANSFORM_OCTAGONAL84, from_border);
    gwy_field_linear_combination(distances, 0.5, distances, 0.5, tmp, 0.0);
    g_object_unref(tmp);
}

/**
 * gwy_field_grain_simple_dist_trans:
 * @nield: A number field with zeros in empty space and nonzeros in marked areas.
 * @distances: Output data field which will be resized and filled with the distances.
 * @dtype: Type of simple distance to use.
 * @from_border: %TRUE to consider image edges to be grain boundaries.
 *
 * Performs a distance transform of a number field.
 *
 * For each non-zero value, a distance to the grain boundary will be calculated, measured in pixels. The function does
 * not distiguish grain numbers, only marked/unmarked areas.
 *
 * If the entire @nield is marked and @from_border is %FALSE the grain has no boundaries. In such case @distances is
 * filled with a huge positive value (like %G_MAXDOUBLE).
 **/
void
gwy_nield_distance_transform(GwyNield *nield,
                             GwyField *distances,
                             GwyDistanceTransformType dtype,
                             gboolean from_border)
{
    g_return_if_fail(GWY_IS_NIELD(nield));
    g_return_if_fail(GWY_IS_FIELD(distances));

    gint xres = nield->xres, yres = nield->yres;
    gwy_field_resize(distances, xres, yres);

    gsize n = (gsize)xres * (gsize)yres;
    gint *g = nield->priv->data;

    if (!from_border) {
        /* Handle the infinite distance case once for all here so that all methods return consitently the same
         * huge number and do not need special-casing. */
        gboolean any_empty = FALSE;
        for (gsize k = 0; k < n; k++) {
            if (g[k] <= 0) {
                any_empty = TRUE;
                break;
            }
        }
        if (!any_empty) {
            gwy_field_fill(distances, G_MAXDOUBLE);
            return;
        }
    }

    if (dtype == GWY_DISTANCE_TRANSFORM_EUCLIDEAN) {
        euclidean_distance_transform(nield, distances, from_border);
        return;
    }
    if (dtype == GWY_DISTANCE_TRANSFORM_OCTAGONAL) {
        average_octagonal_dt(nield, distances, from_border);
        return;
    }

    /* The other distances are always integers. */
    g_return_if_fail(dtype <= GWY_DISTANCE_TRANSFORM_OCTAGONAL84);

    PixelQueue *inqueue = g_new0(PixelQueue, 1);
    PixelQueue *outqueue = g_new0(PixelQueue, 1);
    guint *workspace = g_new(guint, n);
    for (gsize k = 0; k < n; k++)
        workspace[k] = (g[k] > 0) ? G_MAXUINT : 0;

    _gwy_simple_dist_trans(workspace, xres, yres, from_border, dtype, inqueue, outqueue);

    gdouble *d = distances->priv->data;
    for (gsize k = 0; k < n; k++)
        d[k] = workspace[k];
    gwy_field_invalidate(distances);

    g_free(workspace);
    g_free(inqueue->points);
    g_free(outqueue->points);
    g_free(inqueue);
    g_free(outqueue);
}

/**
 * gwy_nield_shrink:
 * @nield: A number field with zeros in empty space and nonzeros in grains.
 * @amount: How much the grains should be reduced, in pixels.  It is inclusive, i.e. pixels that are @amount far from
 *          the border will be removed.
 * @dtype: Type of simple distance to use.
 * @from_border: %TRUE to consider image edges to be grain boundaries. %FALSE to reduce grains touching field
 *               boundaries only along the boundaries.
 *
 * Erodes a number field containing mask by specified amount using a distance measure.
 *
 * Non-zero pixels in @nield will be replaced with zeros if they are not farther than @amount from the grain
 * boundary as defined by @dtype.
 **/
void
gwy_nield_shrink(GwyNield *nield,
                 gdouble amount,
                 GwyDistanceTransformType dtype,
                 gboolean from_border)
{
    g_return_if_fail(GWY_IS_NIELD(nield));
    g_return_if_fail(dtype <= GWY_DISTANCE_TRANSFORM_EUCLIDEAN);

    if (amount < 0.5)
        return;

    amount += 1e-9;
    gint xres = nield->xres, yres = nield->yres;
    if (from_border && amount > MAX(xres/2, yres/2) + 1) {
        gwy_nield_clear(nield);
        return;
    }

    gsize n = (gsize)xres * (gsize)yres;
    GwyField *distances = gwy_field_new(xres, yres, xres, yres, FALSE);
    gwy_nield_distance_transform(nield, distances, dtype, from_border);
    gint *g = nield->priv->data;
    const gdouble *d = distances->priv->data;
    for (gsize k = 0; k < n; k++) {
        if (d[k] <= amount)
            g[k] = 0;
    }

    g_object_unref(distances);
    gwy_nield_invalidate(nield);
}

static gint
compare_distant_points(gconstpointer pa, gconstpointer pb)
{
    const DistantPoint *a = (const DistantPoint*)pa;
    const DistantPoint *b = (const DistantPoint*)pb;

    if (a->distance < b->distance)
        return -1;
    if (a->distance > b->distance)
        return 1;
    if (a->i < b->i)
        return -1;
    if (a->i > b->i)
        return 1;
    if (a->j < b->j)
        return -1;
    if (a->j > b->j)
        return 1;
    return 0;
}

static void
grow_without_merging(GwyNield *nield, GwyField *distances, gdouble amount)
{
    if (!gwy_nield_number_contiguous(nield))
        return;

    gint xres = nield->xres, yres = nield->yres;
    gint *d = nield->priv->data;
    const gdouble *e = distances->priv->data;

    GArray *array = g_array_sized_new(FALSE, FALSE, sizeof(DistantPoint), 1000);
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++) {
            gdouble eij = e[i*xres + j];
            if (eij > 0.0 && eij <= amount) {
                DistantPoint dp = { eij, i, j };
                g_array_append_val(array, dp);
            }
        }
    }
    g_array_sort(array, compare_distant_points);

    for (guint m = 0; m < array->len; m++) {
        const DistantPoint *dp = &g_array_index(array, DistantPoint, m);
        gint k = dp->i*xres + dp->j;
        gint g1 = dp->i > 0      ? d[k-xres] : 0;
        gint g2 = dp->j > 0      ? d[k-1]    : 0;
        gint g3 = dp->j < xres-1 ? d[k+1]    : 0;
        gint g4 = dp->i < yres-1 ? d[k+xres] : 0;
        /* If all are equal or zeros then bitwise or gives us the nonzero value sought. */
        gint gno = g1 | g2 | g3 | g4;
        if ((!g1 || g1 == gno) && (!g2 || g2 == gno) && (!g3 || g3 == gno) && (!g4 || g4 == gno))
            d[k] = gno;
    }

    g_array_free(array, TRUE);
}

/**
 * gwy_nield_grow:
 * @nield: A number field with zeros in empty space and nonzeros in grains.
 * @amount: How much the grains should be expanded, in pixels.  It is inclusive, i.e. exterior pixels that are @amount
 *          far from the border will be filled.
 * @dtype: Type of simple distance to use.
 * @prevent_merging: %TRUE to prevent grain merging, i.e. the growth stops where two grains would merge.  %FALSE to
 *                   simply expand the grains, without regard to grain connectivity.
 *
 * Dilates a number field containing mask by specified amount using a distance measure.
 *
 * Non-positive pixels in @field will be replaced with ones if they are not farther than @amount from the grain
 * boundary as defined by @dtype.
 **/
void
gwy_nield_grow(GwyNield *nield,
               gdouble amount,
               GwyDistanceTransformType dtype,
               gboolean prevent_merging)
{
    g_return_if_fail(GWY_IS_NIELD(nield));
    g_return_if_fail(dtype <= GWY_DISTANCE_TRANSFORM_EUCLIDEAN);

    if (amount < 0.5)
        return;

    amount += 1e-9;
    gint xres = nield->xres, yres = nield->yres;
    gsize n = (gsize)xres * (gsize)yres;
    GwyField *distances = gwy_field_new(xres, yres, xres, yres, FALSE);
    gwy_nield_invert(nield);
    gwy_nield_distance_transform(nield, distances, dtype, FALSE);
    gwy_nield_invert(nield);
    if (prevent_merging)
        grow_without_merging(nield, distances, amount);
    else {
        const gdouble *d = distances->priv->data;
        gint *g = nield->priv->data;
        for (gsize k = 0; k < n; k++) {
            if (d[k] <= amount)
                g[k] = 1.0;
        }
    }
    g_object_unref(distances);
    gwy_nield_invalidate(nield);
}

static gint
compare_candidate(gconstpointer pa, gconstpointer pb)
{
    const ThinCandidate *a = (const ThinCandidate*)pa;
    const ThinCandidate *b = (const ThinCandidate*)pb;

    /* Take pixels with lowest Euclidean distances first. */
    if (a->dist < b->dist)
        return -1;
    if (a->dist > b->dist)
        return 1;

    /* If equal, take pixels with largest Euclidean distance *of their neighbours* first.  This essentially mean flat
     * edges go before corners, preserving useful branches. */
    if (a->ndist > b->ndist)
        return -1;
    if (a->ndist < b->ndist)
        return 1;

    /* When desperate, sort bottom and right coordinates first so that we try to remove them first.  Anyway we must
     * impose some rule to make the sort stable. */
    if (a->i > b->i)
        return -1;
    if (a->i < b->i)
        return 1;
    if (a->j > b->j)
        return -1;
    if (a->j < b->j)
        return 1;

    return 0;
}

/**
 * gwy_nield_thin:
 * @field: A data field with zeros in empty space and nonzeros in grains.
 *
 * Performs thinning of a data field containing mask.
 *
 * The result of thinning is a ‘skeleton’ mask consisting of single-pixel thin lines.
 **/
void
gwy_nield_thin(GwyNield *nield)
{
    /* TRUE means removing the central pixel in a 3x3 pixel configuration does not break any currently connected
     * parts. */
    static const gboolean ok_to_remove[0x100] = {
        FALSE, TRUE,  FALSE, TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  FALSE, TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE,  TRUE,
        TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE,  TRUE,
        FALSE, TRUE,  FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
        TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE,  TRUE,
        TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE,  TRUE,
        TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE,  TRUE,
        TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
        TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE,  TRUE,
        FALSE, TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE,
        TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE,
        FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
        FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE,
        TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE,
        TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE,
        TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, TRUE,
        TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  FALSE, TRUE,  TRUE,  TRUE,  TRUE,  TRUE,
    };

    g_return_if_fail(GWY_IS_NIELD(nield));

    gint xres = nield->xres, yres = nield->yres;
    gsize n = (gsize)xres * (gsize)yres;
    GwyField *distances = gwy_field_new(xres, yres, xres, yres, FALSE);
    gwy_nield_distance_transform(nield, distances, GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE);
    gdouble *d = distances->priv->data;
    gint *g = nield->priv->data;

    gint ncand = 0;
    for (gsize k = 0; k < n; k++) {
        if (d[k] > 0.0)
            ncand++;
    }

    if (ncand < 2) {
        /* There are no mask pixels or just a single pixel.  In either case we do not have to do anything. */
        g_object_unref(distances);
        return;
    }

    ThinCandidate *candidates = g_new(ThinCandidate, ncand);
    gint k = 0;
    for (gint i = 0; i < yres; i++) {
        for (gint j = 0; j < xres; j++) {
            gint m = i*xres + j;
            if (d[m] > 0.0) {
                gdouble nd, ndist = 0.0, maxndist = 0.0;
                candidates[k].i = i;
                candidates[k].j = j;
                candidates[k].dist = d[m];

                if (i && j) {
                    nd = d[m-xres-1];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (i) {
                    nd = d[m-xres];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (i && j < xres-1) {
                    nd = d[m-xres+1];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (j < xres-1) {
                    nd = d[m+1];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (i < yres-1 && j < xres-1) {
                    nd = d[m+xres+1];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (i < yres-1) {
                    nd = d[m+xres];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (i < yres-1 && j) {
                    nd = d[m+xres-1];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }
                if (j) {
                    nd = d[m-1];
                    ndist += nd;
                    if (nd > maxndist)
                        maxndist = nd;
                }

                /* If the point is farther from the border than any neighbour we never remove it. */
                if (candidates[k].dist < 0.999*maxndist) {
                    candidates[k].ndist = ndist;
                    k++;
                }
            }
        }
    }
    ncand = k;

    if (ncand) {
        qsort(candidates, ncand, sizeof(ThinCandidate), &compare_candidate);

        for (k = 0; k < ncand; k++) {
            guint b = 0;

            gint i = candidates[k].i;
            gint j = candidates[k].j;
            if (i && j && d[(i-1)*xres + (j-1)] > 0.0)
                b |= 1;
            if (i && d[(i-1)*xres + j] > 0.0)
                b |= 2;
            if (i && j < xres-1 && d[(i-1)*xres + (j+1)] > 0.0)
                b |= 4;
            if (j < xres-1 && d[i*xres + (j+1)] > 0.0)
                b |= 8;
            if (i < yres-1 && j < xres-1 && d[(i+1)*xres + (j+1)] > 0.0)
                b |= 16;
            if (i < yres-1 && d[(i+1)*xres + j] > 0.0)
                b |= 32;
            if (i < yres-1 && j && d[(i+1)*xres + (j-1)] > 0.0)
                b |= 64;
            if (j && d[i*xres + (j-1)] > 0.0)
                b |= 128;

            if (ok_to_remove[b]) {
                d[i*xres + j] = 0.0;
                g[i*xres + j] = 0;
            }
        }
    }

    g_free(candidates);
    g_object_unref(distances);
    gwy_nield_invalidate(nield);
}

/**
 * gwy_distance_transform_type_get_enum:
 *
 * Returns #GwyEnum for #GwyDistanceTransformType enum type.
 *
 * Returns: %NULL-terminated #GwyEnum which must not be modified nor freed.
 **/
const GwyEnum*
gwy_distance_transform_type_get_enum(void)
{
    static const GwyEnum entries[] = {
        { gwy_NC("distance", "City-block"),    GWY_DISTANCE_TRANSFORM_CITYBLOCK,   },
        { gwy_NC("distance", "Chess"),         GWY_DISTANCE_TRANSFORM_CHESS,       },
        { gwy_NC("distance", "Octagonal 4,8"), GWY_DISTANCE_TRANSFORM_OCTAGONAL48, },
        { gwy_NC("distance", "Octagonal 8,4"), GWY_DISTANCE_TRANSFORM_OCTAGONAL84, },
        { gwy_NC("distance", "Octagonal"),     GWY_DISTANCE_TRANSFORM_OCTAGONAL,   },
        { gwy_NC("distance", "Euclidean"),     GWY_DISTANCE_TRANSFORM_EUCLIDEAN,   },
        { NULL,                                0,                                  },
    };
    return entries;
}

/**
 * SECTION: distance-transform
 * @title: Distance transform
 * @short_description: Distance transform and related morphological operations
 **/

/**
 * GwyDistanceTransformType:
 * @GWY_DISTANCE_TRANSFORM_CITYBLOCK: City-block distance (sum of horizontal and vertical distances).
 * @GWY_DISTANCE_TRANSFORM_CONN4: Four-connectivity distance; another name for city-block distance.
 * @GWY_DISTANCE_TRANSFORM_CHESS: Chessboard distance (maximum of horizontal and vertical distance).
 * @GWY_DISTANCE_TRANSFORM_CONN8: Eight-connectivity distance; another name for chessboard distance.
 * @GWY_DISTANCE_TRANSFORM_OCTAGONAL48: Octagonal distance beginning from city-block.
 * @GWY_DISTANCE_TRANSFORM_OCTAGONAL84: Octagonal distance beginning from chess.
 * @GWY_DISTANCE_TRANSFORM_OCTAGONAL: Average octagonal distance, i.e. the mean of the 48 and 84 distances.
 * @GWY_DISTANCE_TRANSFORM_EUCLIDEAN: True Euclidean distance.
 *
 * Type of distance transform.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
